<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\LoanOffer;

class FixLoanAmounts extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'loans:fix-amounts';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Fix total_amount_to_pay for loan offers where it is 0 or null';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Starting to fix loan amounts...');
        
        // Get all loan offers where total_amount_to_pay is 0 or null
        $loanOffers = LoanOffer::where(function($query) {
            $query->whereNull('total_amount_to_pay')
                  ->orWhere('total_amount_to_pay', 0);
        })->get();
        
        $totalCount = $loanOffers->count();
        $updatedCount = 0;
        
        if ($totalCount == 0) {
            $this->info('No loan offers need fixing for total_amount_to_pay.');
        } else {
            $this->info("Found {$totalCount} loan offers with missing total_amount_to_pay.");
            
            foreach ($loanOffers as $loanOffer) {
                // Calculate total_amount_to_pay based on loan terms
                // This is typically the principal + interest over the tenure
                $principal = $loanOffer->requested_amount ?? 0;
                $interestRate = $loanOffer->interest_rate ?? 12; // Default 12% if not set
                $tenure = $loanOffer->tenure ?? 12; // Default 12 months if not set
                
                // Calculate monthly interest rate
                $monthlyRate = $interestRate / 100 / 12;
                
                // Calculate total amount to pay using compound interest formula
                // For simple calculation: Total = Monthly Payment * Number of Months
                if ($monthlyRate > 0 && $principal > 0 && $tenure > 0) {
                    // Calculate monthly payment using amortization formula
                    $monthlyPayment = ($principal * $monthlyRate * pow(1 + $monthlyRate, $tenure)) / 
                                     (pow(1 + $monthlyRate, $tenure) - 1);
                    
                    $totalAmountToPay = $monthlyPayment * $tenure;
                } else {
                    // Fallback: simple interest calculation
                    $totalInterest = $principal * ($interestRate / 100) * ($tenure / 12);
                    $totalAmountToPay = $principal + $totalInterest;
                }
                
                $loanOffer->total_amount_to_pay = $totalAmountToPay;
                $loanOffer->save();
                
                $updatedCount++;
                $this->line("Updated Loan #{$loanOffer->id}: Total to Pay = " . number_format($totalAmountToPay, 2));
            }
            
            $this->info("Successfully updated {$updatedCount} loan offers with total_amount_to_pay.");
        }
        
        // Also ensure take_home_amount is calculated correctly with all charges
        $this->info("\nVerifying take_home_amount calculations include all charges...");
        
        $loanOffers = LoanOffer::all();
        $recalculatedCount = 0;
        
        foreach ($loanOffers as $loanOffer) {
            $principal = $loanOffer->requested_amount ?? 0;
            $processingFee = $loanOffer->processing_fee ?? 0;
            $insurance = $loanOffer->insurance ?? 0;
            $otherCharges = $loanOffer->other_charges ?? 0;
            
            // Total charges include processing fee, insurance, and other charges
            $totalCharges = $processingFee + $insurance + $otherCharges;
            
            // Take home amount is principal minus all charges
            $correctTakeHome = $principal - $totalCharges;
            
            // Check if take_home_amount needs updating
            if (abs($loanOffer->take_home_amount - $correctTakeHome) > 0.01) {
                $this->line("Loan #{$loanOffer->id}: Updating take_home from {$loanOffer->take_home_amount} to {$correctTakeHome}");
                $this->line("  Principal: {$principal}, Processing: {$processingFee}, Insurance: {$insurance}, Other: {$otherCharges}");
                
                $loanOffer->take_home_amount = $correctTakeHome;
                $loanOffer->save();
                $recalculatedCount++;
            }
        }
        
        if ($recalculatedCount > 0) {
            $this->info("Recalculated take_home_amount for {$recalculatedCount} loan offers.");
        } else {
            $this->info("All take_home_amount values are correctly calculated.");
        }
        
        // Display summary
        $this->line("\n=== Summary ===");
        $stats = \DB::table('loan_offers')
            ->selectRaw('
                COUNT(*) as total_records,
                AVG(requested_amount) as avg_requested,
                AVG(total_amount_to_pay) as avg_total_to_pay,
                AVG(take_home_amount) as avg_take_home,
                AVG(processing_fee) as avg_processing,
                AVG(insurance) as avg_insurance,
                AVG(other_charges) as avg_other
            ')
            ->first();
        
        $this->table(
            ['Metric', 'Average Value'],
            [
                ['Total Records', number_format($stats->total_records)],
                ['Requested Amount', 'TZS ' . number_format($stats->avg_requested, 2)],
                ['Total to Pay', 'TZS ' . number_format($stats->avg_total_to_pay, 2)],
                ['Take Home Amount', 'TZS ' . number_format($stats->avg_take_home, 2)],
                ['Processing Fee', 'TZS ' . number_format($stats->avg_processing, 2)],
                ['Insurance', 'TZS ' . number_format($stats->avg_insurance, 2)],
                ['Other Charges', 'TZS ' . number_format($stats->avg_other, 2)],
            ]
        );
        
        return Command::SUCCESS;
    }
}